<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Notifications\BroadcastNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Inertia\Inertia;

class BroadcastController extends Controller
{
    /**
     * Show the broadcast form.
     */
    public function index()
    {
        $coachesCount = User::where('role', 'coach')->where('status', 'active')->count();
        
        $recentBroadcasts = \App\Models\Setting::where('key', 'like', 'broadcast_history_%')
            ->orderBy('updated_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($setting) {
                $data = json_decode($setting->value, true);
                return [
                    'id' => $setting->id,
                    'subject' => $data['subject'] ?? '',
                    'message' => $data['message'] ?? '',
                    'sent_to' => $data['sent_to'] ?? 0,
                    'sent_at' => $setting->updated_at,
                ];
            });

        return Inertia::render('Admin/Broadcast/Index', [
            'coachesCount' => $coachesCount,
            'recentBroadcasts' => $recentBroadcasts,
        ]);
    }

    /**
     * Send broadcast notification to coaches.
     */
    public function send(Request $request)
    {
        $validated = $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
            'send_email' => 'boolean',
            'send_database' => 'boolean',
            'target' => 'required|in:all,active,pending,suspended',
        ]);

        // Validate at least one channel is selected
        if (!($validated['send_email'] ?? false) && !($validated['send_database'] ?? true)) {
            return back()->withErrors(['channels' => 'Debes seleccionar al menos un canal de envío.']);
        }

        // Build query based on target
        $query = User::where('role', 'coach');
        
        switch ($validated['target']) {
            case 'active':
                $query->where('status', 'active');
                break;
            case 'pending':
                $query->where('status', 'pending');
                break;
            case 'suspended':
                $query->where('status', 'suspended');
                break;
            // 'all' - no filter needed
        }

        $coaches = $query->get();
        $count = $coaches->count();

        if ($count === 0) {
            return back()->with('error', 'No hay entrenadores que coincidan con los criterios seleccionados.');
        }

        // Determine channels
        $channels = [];
        if ($validated['send_database'] ?? true) {
            $channels[] = 'database';
        }
        if ($validated['send_email'] ?? false) {
            $channels[] = 'mail';
        }

        // Send notification
        Notification::send($coaches, new BroadcastNotification(
            $validated['subject'],
            $validated['message'],
            $channels
        ));

        // Save to history
        \App\Models\Setting::set(
            'broadcast_history_' . now()->timestamp,
            json_encode([
                'subject' => $validated['subject'],
                'message' => $validated['message'],
                'sent_to' => $count,
                'target' => $validated['target'],
                'channels' => $channels,
                'sent_by' => auth()->id(),
            ]),
            'broadcast'
        );

        return back()->with('success', "Notificación enviada exitosamente a {$count} entrenadores.");
    }
}
