<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\Plan;
use App\Models\Setting;
use App\Models\Subscription;
use App\Models\SubscriptionAuditLog;
use App\Models\User;
use App\Notifications\InvoicePaid;
use App\Notifications\SubscriptionExtended;
use App\Notifications\SubscriptionPlanChanged;
use App\Notifications\SubscriptionStatusChanged;
use App\Services\PdfBrowsershotService;
use App\Services\PdfService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Inertia\Response;

class SubscriptionController extends Controller
{
    protected PdfService $pdfService;

    public function __construct(PdfService $pdfService)
    {
        $this->pdfService = $pdfService;
    }

    /**
     * Display a listing of subscriptions.
     */
    public function index(Request $request): Response
    {
        $query = Subscription::with(['user', 'plan', 'invoice'])
            ->latest();

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('plan')) {
            $query->where('plan_id', $request->plan);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $subscriptions = $query->paginate(15)->withQueryString();

        // Get stats
        $stats = [
            'total' => Subscription::count(),
            'active' => Subscription::where('status', 'active')->count(),
            'pending' => Subscription::where('status', 'pending')->count(),
            'expired' => Subscription::where('status', 'expired')->count(),
            'cancelled' => Subscription::where('status', 'cancelled')->count(),
            'revenue_this_month' => Invoice::where('status', 'paid')
                ->whereMonth('paid_at', now()->month)
                ->whereYear('paid_at', now()->year)
                ->sum('total'),
        ];

        // Get active payment methods
        $paymentMethods = $this->getActivePaymentMethods();

        return Inertia::render('Admin/Subscriptions/Index', [
            'subscriptions' => $subscriptions,
            'plans' => Plan::where('is_active', true)->get(),
            'stats' => $stats,
            'filters' => $request->only(['status', 'plan', 'search', 'date_from', 'date_to']),
            'paymentMethods' => $paymentMethods,
        ]);
    }

    /**
     * Get active payment methods based on settings.
     */
    protected function getActivePaymentMethods(): array
    {
        $methods = [];

        if (\App\Models\Setting::get('bank_transfer_enabled', true)) {
            $methods[] = ['id' => 'bank_transfer', 'name' => 'Transferencia Bancaria'];
        }
        if (\App\Models\Setting::get('paypal_enabled', false)) {
            $methods[] = ['id' => 'paypal', 'name' => 'PayPal'];
        }
        if (\App\Models\Setting::get('stripe_enabled', false)) {
            $methods[] = ['id' => 'stripe', 'name' => 'Stripe'];
        }

        return $methods;
    }

    /**
     * Show subscription details.
     */
    public function show(Subscription $subscription): Response
    {
        $subscription->load(['user', 'plan', 'invoice.selectedBank', 'payments', 'auditLogs.admin']);

        // Get active payment methods
        $paymentMethods = $this->getActivePaymentMethods();

        return Inertia::render('Admin/Subscriptions/Show', [
            'subscription' => $subscription,
            'plans' => Plan::where('is_active', true)->get(),
            'paymentMethods' => $paymentMethods,
        ]);
    }

    /**
     * Update subscription status.
     */
    public function updateStatus(Request $request, Subscription $subscription): RedirectResponse
    {
        $validated = $request->validate([
            'status' => 'required|in:active,pending,expired,cancelled,suspended',
            'notes' => 'nullable|string|max:500',
        ]);

        $oldStatus = $subscription->status;
        $newStatus = $validated['status'];

        if ($oldStatus === $newStatus) {
            return back()->with('info', 'El estado no ha cambiado.');
        }

        DB::transaction(function () use ($subscription, $oldStatus, $newStatus, $validated) {
            // Update subscription
            $subscription->update([
                'status' => $newStatus,
            ]);

            // Handle activation
            if ($newStatus === 'active' && $oldStatus !== 'active') {
                // Set dates if activating for the first time
                if (!$subscription->start_date || $subscription->start_date->isPast()) {
                    $subscription->update([
                        'start_date' => now(),
                        'end_date' => now()->addDays($subscription->plan->duration_days),
                    ]);
                }
            }

            // Log the change
            SubscriptionAuditLog::log(
                $subscription->id,
                SubscriptionAuditLog::ACTION_STATUS_CHANGED,
                $oldStatus,
                $newStatus,
                ['reason' => $validated['notes'] ?? 'Status updated by admin'],
                $validated['notes'] ?? null
            );
        });

        // Notify coach about status change
        $subscription->user->notify(new SubscriptionStatusChanged(
            $subscription,
            $oldStatus,
            $newStatus,
            $validated['notes'] ?? null
        ));

        return back()->with('success', "Estado actualizado a '{$newStatus}'.");
    }

    /**
     * Confirm payment manually (for bank transfers).
     */
    public function confirmPayment(Request $request, Subscription $subscription): RedirectResponse
    {
        $validated = $request->validate([
            'payment_method' => 'required|in:bank_transfer,cash,other',
            'transaction_reference' => 'nullable|string|max:255',
            'notes' => 'nullable|string|max:500',
        ]);

        DB::transaction(function () use ($subscription, $validated) {
            // Get or create invoice
            $invoice = $subscription->invoice;

            if (!$invoice) {
                // Create invoice if doesn't exist
                $invoice = Invoice::create([
                    'user_id' => $subscription->user_id,
                    'subscription_id' => $subscription->id,
                    'invoice_number' => Invoice::generateNumber(),
                    'subtotal' => $subscription->plan->price,
                    'tax' => 0,
                    'total' => $subscription->plan->price,
                    'status' => 'pending',
                    'items' => json_encode([
                        [
                            'description' => "Plan {$subscription->plan->name}",
                            'quantity' => 1,
                            'unit_price' => $subscription->plan->price,
                            'total' => $subscription->plan->price,
                        ]
                    ]),
                ]);
            }

            // Update invoice
            $invoice->update([
                'status' => 'paid',
                'payment_method' => $validated['payment_method'],
                'transaction_id' => $validated['transaction_reference'],
                'paid_at' => now(),
                'notes' => $validated['notes'],
            ]);

            // Create or update payment record
            $payment = Payment::updateOrCreate(
                ['invoice_id' => $invoice->id],
                [
                    'user_id' => $subscription->user_id,
                    'subscription_id' => $subscription->id,
                    'gateway' => $validated['payment_method'],
                    'amount' => $invoice->total,
                    'currency' => 'USD',
                    'status' => 'completed',
                    'gateway_payment_id' => $validated['transaction_reference'],
                    'paid_at' => now(),
                ]
            );

            // Generate PDF if not exists
            $this->generateInvoicePdf($invoice);

            // Activate subscription
            $subscription->update([
                'status' => 'active',
                'start_date' => now(),
                'end_date' => now()->addDays($subscription->plan->duration_days),
            ]);

            // Expire other active subscriptions
            Subscription::where('user_id', $subscription->user_id)
                ->where('id', '!=', $subscription->id)
                ->where('status', 'active')
                ->update(['status' => 'expired']);

            // Log the confirmation
            SubscriptionAuditLog::log(
                $subscription->id,
                SubscriptionAuditLog::ACTION_PAYMENT_CONFIRMED,
                'pending',
                'paid',
                [
                    'payment_method' => $validated['payment_method'],
                    'reference' => $validated['transaction_reference'],
                    'amount' => $invoice->total,
                ],
                $validated['notes']
            );

            // Notify user
            $subscription->user->notify(new InvoicePaid($invoice));
        });

        return back()->with('success', 'Pago confirmado y suscripción activada.');
    }

    /**
     * Change subscription plan.
     */
    public function changePlan(Request $request, Subscription $subscription): RedirectResponse
    {
        $validated = $request->validate([
            'plan_id' => 'required|exists:plans,id',
            'extend_days' => 'nullable|boolean',
            'notes' => 'nullable|string|max:500',
        ]);

        $oldPlan = $subscription->plan;
        $newPlan = Plan::findOrFail($validated['plan_id']);

        if ($oldPlan->id === $newPlan->id) {
            return back()->with('info', 'El plan seleccionado es el mismo.');
        }

        DB::transaction(function () use ($subscription, $oldPlan, $newPlan, $validated) {
            $newEndDate = $subscription->end_date;

            // Optionally extend the subscription
            if ($validated['extend_days'] ?? false) {
                $newEndDate = now()->addDays($newPlan->duration_days);
            }

            $subscription->update([
                'plan_id' => $newPlan->id,
                'end_date' => $newEndDate,
            ]);

            // Log the change
            SubscriptionAuditLog::log(
                $subscription->id,
                SubscriptionAuditLog::ACTION_PLAN_CHANGED,
                $oldPlan->name,
                $newPlan->name,
                [
                    'old_plan_id' => $oldPlan->id,
                    'new_plan_id' => $newPlan->id,
                    'extended' => $validated['extend_days'] ?? false,
                ],
                $validated['notes']
            );
        });

        // Notify coach about plan change
        $subscription->user->notify(new SubscriptionPlanChanged(
            $subscription,
            $oldPlan->name,
            $newPlan->name
        ));

        return back()->with('success', "Plan cambiado a '{$newPlan->name}'.");
    }

    /**
     * Extend subscription.
     */
    public function extend(Request $request, Subscription $subscription): RedirectResponse
    {
        $validated = $request->validate([
            'days' => 'required|integer|min:1|max:365',
            'notes' => 'nullable|string|max:500',
        ]);

        $oldEndDate = $subscription->end_date;

        // Extend from current end date or from now if already expired
        $baseDate = $subscription->end_date && $subscription->end_date->isFuture()
            ? $subscription->end_date
            : now();

        $newEndDate = $baseDate->copy()->addDays($validated['days']);

        DB::transaction(function () use ($subscription, $oldEndDate, $newEndDate, $validated) {
            $subscription->update([
                'end_date' => $newEndDate,
                'status' => 'active', // Reactivate if expired
            ]);

            // Log the change
            SubscriptionAuditLog::log(
                $subscription->id,
                SubscriptionAuditLog::ACTION_EXTENDED,
                $oldEndDate?->format('Y-m-d'),
                $newEndDate->format('Y-m-d'),
                ['days_added' => $validated['days']],
                $validated['notes']
            );
        });

        // Notify coach about extension
        $subscription->user->notify(new SubscriptionExtended(
            $subscription->fresh(),
            $validated['days']
        ));

        return back()->with('success', "Suscripción extendida por {$validated['days']} días.");
    }

    /**
     * Cancel subscription.
     */
    public function cancel(Request $request, Subscription $subscription): RedirectResponse
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:500',
            'refund' => 'nullable|boolean',
        ]);

        DB::transaction(function () use ($subscription, $validated) {
            $subscription->update([
                'status' => 'cancelled',
            ]);

            // Log the cancellation
            SubscriptionAuditLog::log(
                $subscription->id,
                SubscriptionAuditLog::ACTION_CANCELLED,
                $subscription->status,
                'cancelled',
                ['refund_requested' => $validated['refund'] ?? false],
                $validated['reason']
            );
        });

        // Notify coach about cancellation
        $subscription->user->notify(new SubscriptionStatusChanged(
            $subscription,
            $subscription->status,
            'cancelled',
            $validated['reason']
        ));

        return back()->with('success', 'Suscripción cancelada.');
    }

    /**
     * Get audit logs for a subscription.
     */
    public function auditLogs(Subscription $subscription): Response
    {
        $logs = $subscription->auditLogs()
            ->with('admin')
            ->latest()
            ->paginate(20);

        return Inertia::render('Admin/Subscriptions/AuditLogs', [
            'subscription' => $subscription->load(['user', 'plan']),
            'logs' => $logs,
        ]);
    }

    /**
     * Download invoice PDF.
     */
    public function downloadInvoice(Invoice $invoice)
    {
        $invoice->load(['user', 'subscription.plan']);

        try {
            $filename = $this->generateInvoicePdf($invoice);
            $filePath = storage_path("app/public/{$filename}");

            if (!file_exists($filePath)) {
                abort(404, 'No se pudo generar el PDF de la factura.');
            }

            return response()->download($filePath, "factura-{$invoice->invoice_number}.pdf");
        } catch (\Exception $e) {
            Log::error('Admin Subscription Invoice PDF Error: ' . $e->getMessage());
            $filename = $this->pdfService->generateInvoice($invoice);
            $filePath = storage_path("app/public/{$filename}");
            return response()->download($filePath, "factura-{$invoice->invoice_number}.pdf");
        }
    }

    /**
     * Generate invoice PDF using the configured engine.
     */
    protected function generateInvoicePdf(Invoice $invoice): string
    {
        try {
            $pdfEngine = Setting::get('pdf_engine', 'browsershot');

            if ($pdfEngine === 'browsershot') {
                return app(PdfBrowsershotService::class)->generateInvoice($invoice);
            }

            return $this->pdfService->generateInvoice($invoice);
        } catch (\Exception $e) {
            Log::warning('Browsershot failed, falling back to DomPDF: ' . $e->getMessage());
            return $this->pdfService->generateInvoice($invoice);
        }
    }
}
