<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LandingTestimonial;
use App\Notifications\TestimonialApproved;
use App\Notifications\TestimonialRejected;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TestimonialApprovalController extends Controller
{
    /**
     * Display testimonials management page with pending reviews.
     */
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'pending');

        $query = LandingTestimonial::with('coach:id,name,email')
            ->orderBy('submitted_at', 'desc');

        if ($filter === 'pending') {
            $query->where('status', LandingTestimonial::STATUS_PENDING);
        } elseif ($filter === 'approved') {
            $query->where('status', LandingTestimonial::STATUS_APPROVED);
        } elseif ($filter === 'rejected') {
            $query->where('status', LandingTestimonial::STATUS_REJECTED);
        } elseif ($filter === 'coach') {
            $query->where('source', LandingTestimonial::SOURCE_COACH);
        }

        $testimonials = $query->get()->map(function ($testimonial) {
            return [
                'id' => $testimonial->id,
                'name' => $testimonial->name,
                'role' => $testimonial->role,
                'content' => $testimonial->content,
                'avatar' => $testimonial->avatar,
                'rating' => $testimonial->rating,
                'status' => $testimonial->status,
                'source' => $testimonial->source,
                'rejection_reason' => $testimonial->rejection_reason,
                'is_active' => $testimonial->is_active,
                'sort_order' => $testimonial->sort_order,
                'submitted_at' => $testimonial->submitted_at?->format('d/m/Y H:i'),
                'reviewed_at' => $testimonial->reviewed_at?->format('d/m/Y H:i'),
                'coach' => $testimonial->coach ? [
                    'id' => $testimonial->coach->id,
                    'name' => $testimonial->coach->name,
                    'email' => $testimonial->coach->email,
                ] : null,
            ];
        });

        $counts = [
            'pending' => LandingTestimonial::where('status', LandingTestimonial::STATUS_PENDING)->count(),
            'approved' => LandingTestimonial::where('status', LandingTestimonial::STATUS_APPROVED)->count(),
            'rejected' => LandingTestimonial::where('status', LandingTestimonial::STATUS_REJECTED)->count(),
            'coach_submissions' => LandingTestimonial::where('source', LandingTestimonial::SOURCE_COACH)->count(),
        ];

        return Inertia::render('Admin/Testimonials/Approval', [
            'testimonials' => $testimonials,
            'filter' => $filter,
            'counts' => $counts,
        ]);
    }

    /**
     * Approve a testimonial.
     */
    public function approve(Request $request, LandingTestimonial $testimonial)
    {
        $admin = $request->user();

        $testimonial->approve($admin->id);

        // Notify coach if it was submitted by them
        if ($testimonial->coach) {
            $testimonial->coach->notify(new TestimonialApproved($testimonial));
        }

        return back()->with('success', 'Testimonio aprobado y publicado.');
    }

    /**
     * Reject a testimonial.
     */
    public function reject(Request $request, LandingTestimonial $testimonial)
    {
        $request->validate([
            'reason' => 'nullable|string|max:500',
        ]);

        $admin = $request->user();
        $reason = $request->reason;

        $testimonial->reject($admin->id, $reason);

        // Notify coach if it was submitted by them
        if ($testimonial->coach) {
            $testimonial->coach->notify(new TestimonialRejected($testimonial, $reason));
        }

        return back()->with('success', 'Testimonio rechazado.');
    }

    /**
     * Toggle testimonial active status.
     */
    public function toggleActive(LandingTestimonial $testimonial)
    {
        $testimonial->update([
            'is_active' => !$testimonial->is_active,
        ]);

        $message = $testimonial->is_active 
            ? 'Testimonio visible en la landing.' 
            : 'Testimonio oculto de la landing.';

        return back()->with('success', $message);
    }

    /**
     * Update sort order for testimonials.
     */
    public function reorder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:landing_testimonials,id',
            'items.*.sort_order' => 'required|integer',
        ]);

        foreach ($request->items as $item) {
            LandingTestimonial::where('id', $item['id'])
                ->update(['sort_order' => $item['sort_order']]);
        }

        return back()->with('success', 'Orden actualizado.');
    }

    /**
     * Delete a testimonial.
     */
    public function destroy(LandingTestimonial $testimonial)
    {
        $testimonial->delete();

        return back()->with('success', 'Testimonio eliminado.');
    }
}
