<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\ExerciseLibrary;
use Illuminate\Http\Request;
use Inertia\Inertia;

/**
 * ExerciseController
 * Manages coach's exercise library
 */
class ExerciseController extends Controller
{
    /**
     * Display a listing of exercises
     */
    public function index(Request $request)
    {
        $coach = auth()->user();
        
        $query = ExerciseLibrary::query()
            ->forCoach($coach->id)
            ->active();
        
        // Search filter
        if ($request->search) {
            $query->where('name', 'like', "%{$request->search}%");
        }
        
        // Category filter
        if ($request->category) {
            $query->where('category', $request->category);
        }
        
        // Muscle group filter
        if ($request->muscle_group) {
            $query->where('muscle_group', $request->muscle_group);
        }
        
        $exercises = $query->orderBy('name')->paginate(20)->withQueryString();
        
        return Inertia::render('Coach/Exercises/Index', [
            'exercises' => $exercises,
            'filters' => $request->only(['search', 'category', 'muscle_group']),
            'categories' => ExerciseLibrary::CATEGORIES,
            'muscleGroups' => ExerciseLibrary::MUSCLE_GROUPS,
        ]);
    }
    
    /**
     * Show form for creating a new exercise
     */
    public function create()
    {
        return Inertia::render('Coach/Exercises/Create', [
            'categories' => ExerciseLibrary::CATEGORIES,
            'muscleGroups' => ExerciseLibrary::MUSCLE_GROUPS,
            'equipment' => ExerciseLibrary::EQUIPMENT,
            'difficultyLevels' => ExerciseLibrary::DIFFICULTY_LEVELS,
        ]);
    }
    
    /**
     * Store a newly created exercise
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'category' => 'nullable|string|max:100',
            'muscle_group' => 'nullable|string|max:100',
            'secondary_muscles' => 'nullable|array',
            'secondary_muscles.*' => 'string|max:100',
            'difficulty' => 'nullable|in:principiante,intermedio,avanzado',
            'is_compound' => 'boolean',
            'equipment' => 'nullable|string|max:100',
            'instructions' => 'nullable|string',
            'video_url' => 'nullable|url|max:500',
            'gif_url' => 'nullable|url|max:500',
            'media_file' => 'nullable|file|mimes:gif,jpg,jpeg,png,svg,webp,mp4,webm|max:10240',
        ]);
        
        $validated['coach_id'] = auth()->id();
        $validated['is_compound'] = $request->boolean('is_compound');
        
        // Handle file upload
        if ($request->hasFile('media_file')) {
            $file = $request->file('media_file');
            $filename = 'exercises/' . auth()->id() . '_' . time() . '.' . $file->getClientOriginalExtension();
            $file->storeAs('public', $filename);
            $validated['gif_url'] = '/storage/' . $filename;
        }
        
        // Remove media_file from validated data (not a database field)
        unset($validated['media_file']);
        
        ExerciseLibrary::create($validated);
        
        return redirect()->route('coach.exercises.index')
            ->with('success', 'Ejercicio creado correctamente');
    }
    
    /**
     * Show form for editing an exercise
     */
    public function edit(ExerciseLibrary $exercise_library)
    {
        // Only allow editing own exercises
        if ($exercise_library->coach_id && $exercise_library->coach_id !== auth()->id()) {
            abort(403);
        }
        
        return Inertia::render('Coach/Exercises/Edit', [
            'exercise' => $exercise_library,
            'categories' => ExerciseLibrary::CATEGORIES,
            'muscleGroups' => ExerciseLibrary::MUSCLE_GROUPS,
            'equipment' => ExerciseLibrary::EQUIPMENT,
            'difficultyLevels' => ExerciseLibrary::DIFFICULTY_LEVELS,
        ]);
    }
    
    /**
     * Update the specified exercise
     */
    public function update(Request $request, ExerciseLibrary $exercise_library)
    {
        // Only allow updating own exercises
        if ($exercise_library->coach_id && $exercise_library->coach_id !== auth()->id()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'category' => 'nullable|string|max:100',
            'muscle_group' => 'nullable|string|max:100',
            'secondary_muscles' => 'nullable|array',
            'secondary_muscles.*' => 'string|max:100',
            'difficulty' => 'nullable|in:principiante,intermedio,avanzado',
            'is_compound' => 'boolean',
            'equipment' => 'nullable|string|max:100',
            'instructions' => 'nullable|string',
            'video_url' => 'nullable|url|max:500',
            'gif_url' => 'nullable|url|max:500',
        ]);
        
        $validated['is_compound'] = $request->boolean('is_compound');
        
        $exercise_library->update($validated);
        
        return redirect()->route('coach.exercises.index')
            ->with('success', 'Ejercicio actualizado correctamente');
    }
    
    /**
     * Remove the specified exercise
     */
    public function destroy(ExerciseLibrary $exercise_library)
    {
        // Only allow deleting own exercises
        if (!$exercise_library->coach_id || $exercise_library->coach_id !== auth()->id()) {
            abort(403, 'No puedes eliminar ejercicios del sistema');
        }
        
        $exercise_library->delete();
        
        return back()->with('success', 'Ejercicio eliminado');
    }
}
