<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ExerciseLibrary extends Model
{
    use HasFactory;

    protected $table = 'exercise_library';

    protected $fillable = [
        'coach_id',
        'name',
        'description',
        'category',
        'muscle_group',
        'secondary_muscles',
        'difficulty',
        'is_compound',
        'equipment',
        'instructions',
        'video_url',
        'gif_url',
        'image_url',
        'is_active',
        'is_system',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_compound' => 'boolean',
        'is_system' => 'boolean',
        'secondary_muscles' => 'array',
    ];
    
    /**
     * Niveles de dificultad
     */
    public const DIFFICULTY_LEVELS = [
        'principiante' => 'Principiante',
        'intermedio' => 'Intermedio',
        'avanzado' => 'Avanzado',
    ];

    /**
     * Categorías de ejercicios
     */
    public const CATEGORIES = [
        'fuerza' => 'Fuerza',
        'cardio' => 'Cardio',
        'flexibilidad' => 'Flexibilidad',
        'equilibrio' => 'Equilibrio',
        'funcional' => 'Funcional',
        'hiit' => 'HIIT',
        'pliometria' => 'Pliometría',
        'calistenia' => 'Calistenia',
    ];

    /**
     * Grupos musculares
     */
    public const MUSCLE_GROUPS = [
        'pecho' => 'Pecho',
        'espalda' => 'Espalda',
        'hombros' => 'Hombros',
        'biceps' => 'Bíceps',
        'triceps' => 'Tríceps',
        'antebrazos' => 'Antebrazos',
        'abdominales' => 'Abdominales',
        'oblicuos' => 'Oblicuos',
        'cuadriceps' => 'Cuádriceps',
        'isquiotibiales' => 'Isquiotibiales',
        'gluteos' => 'Glúteos',
        'pantorrillas' => 'Pantorrillas',
        'aductores' => 'Aductores',
        'trapecio' => 'Trapecio',
        'cuerpo_completo' => 'Cuerpo Completo',
    ];

    /**
     * Equipamiento
     */
    public const EQUIPMENT = [
        'ninguno' => 'Sin Equipamiento',
        'mancuernas' => 'Mancuernas',
        'barra' => 'Barra',
        'kettlebell' => 'Kettlebell',
        'bandas' => 'Bandas Elásticas',
        'maquina' => 'Máquina',
        'cable' => 'Poleas/Cable',
        'banca' => 'Banca',
        'barra_dominadas' => 'Barra de Dominadas',
        'pelota' => 'Pelota de Ejercicio',
        'cuerda' => 'Cuerda de Saltar',
        'trx' => 'TRX',
        'bosu' => 'Bosu',
    ];

    /**
     * Coach que creó el ejercicio (null = ejercicio del sistema)
     */
    public function coach(): BelongsTo
    {
        return $this->belongsTo(User::class, 'coach_id');
    }

    /**
     * Scope para ejercicios del sistema (globales)
     */
    public function scopeGlobal($query)
    {
        return $query->whereNull('coach_id');
    }

    /**
     * Scope para ejercicios de un coach específico
     */
    public function scopeForCoach($query, $coachId)
    {
        return $query->where(function ($q) use ($coachId) {
            $q->whereNull('coach_id')
              ->orWhere('coach_id', $coachId);
        });
    }

    /**
     * Scope para ejercicios activos
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
}
