<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PaymentFeeSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'gateway',
        'percentage',
        'fixed_amount',
        'is_active',
        'pass_to_customer',
    ];

    protected $casts = [
        'percentage' => 'decimal:2',
        'fixed_amount' => 'decimal:2',
        'is_active' => 'boolean',
        'pass_to_customer' => 'boolean',
    ];

    /**
     * Calculate the fee for a given amount.
     *
     * @param float $amount The order subtotal
     * @return float The calculated fee
     */
    public function calculateFee(float $amount): float
    {
        $percentageFee = ($amount * $this->percentage) / 100;
        $totalFee = $percentageFee + $this->fixed_amount;
        
        return round($totalFee, 2);
    }

    /**
     * Get fee description for display.
     *
     * @return string
     */
    public function getFeeDescriptionAttribute(): string
    {
        return "{$this->percentage}% + \${$this->fixed_amount}";
    }

    /**
     * Get gateway display name.
     *
     * @return string
     */
    public function getGatewayNameAttribute(): string
    {
        return match($this->gateway) {
            'paypal' => 'PayPal',
            'stripe' => 'Stripe',
            default => ucfirst($this->gateway),
        };
    }

    /**
     * Scope for active settings.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope by gateway.
     */
    public function scopeByGateway($query, string $gateway)
    {
        return $query->where('gateway', $gateway);
    }

    /**
     * Get fee setting for a specific gateway.
     *
     * @param string $gateway
     * @return PaymentFeeSetting|null
     */
    public static function getForGateway(string $gateway): ?PaymentFeeSetting
    {
        return static::active()->byGateway($gateway)->first();
    }
}
