<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class PdfSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'document_type',
        'logo',
        'primary_color',
        'secondary_color',
        'accent_color',
        'header_title',
        'header_subtitle',
        'show_logo',
        'show_date',
        'show_page_numbers',
        'footer_text',
        'footer_note',
        'footer_website',
        'footer_email',
        'footer_phone',
        'show_coach_info',
        'show_coach_photo',
        'show_coach_credentials',
        'show_coach_contact',
        'paper_size',
        'orientation',
        'margin_top',
        'margin_bottom',
        'margin_left',
        'margin_right',
        'font_family',
        'font_size_base',
        'font_size_header',
        'custom_css',
        'watermark_text',
        'watermark_enabled',
        'is_active',
    ];

    protected $casts = [
        'show_logo' => 'boolean',
        'show_date' => 'boolean',
        'show_page_numbers' => 'boolean',
        'show_coach_info' => 'boolean',
        'show_coach_photo' => 'boolean',
        'show_coach_credentials' => 'boolean',
        'show_coach_contact' => 'boolean',
        'watermark_enabled' => 'boolean',
        'is_active' => 'boolean',
        'margin_top' => 'integer',
        'margin_bottom' => 'integer',
        'margin_left' => 'integer',
        'margin_right' => 'integer',
        'font_size_base' => 'integer',
        'font_size_header' => 'integer',
    ];

    /**
     * Document type constants
     */
    const TYPE_INVOICE = 'invoice';
    const TYPE_WORKOUT = 'workout';
    const TYPE_MEAL_PLAN = 'meal_plan';
    const TYPE_STATISTICS = 'statistics';

    /**
     * Get all available document types
     */
    public static function getDocumentTypes(): array
    {
        return [
            self::TYPE_INVOICE => 'Facturas',
            self::TYPE_WORKOUT => 'Rutinas de Ejercicio',
            self::TYPE_MEAL_PLAN => 'Planes de Alimentación',
            self::TYPE_STATISTICS => 'Estadísticas',
        ];
    }

    /**
     * Get settings for a specific document type with caching
     */
    public static function getForType(string $type): ?self
    {
        return Cache::remember("pdf_settings_{$type}", 3600, function () use ($type) {
            return self::where('document_type', $type)
                ->where('is_active', true)
                ->first();
        });
    }

    /**
     * Get settings with fallback to defaults
     */
    public static function getSettingsForType(string $type): array
    {
        $settings = self::getForType($type);
        
        if ($settings) {
            return $settings->toArray();
        }

        // Return default settings if none exist
        return self::getDefaultSettings($type);
    }

    /**
     * Get default settings for a document type
     */
    public static function getDefaultSettings(string $type): array
    {
        $defaults = [
            'document_type' => $type,
            'logo' => null,
            'primary_color' => '#3B82F6',
            'secondary_color' => '#F97316',
            'accent_color' => '#10B981',
            'header_title' => null,
            'header_subtitle' => null,
            'show_logo' => true,
            'show_date' => true,
            'show_page_numbers' => true,
            'footer_text' => null,
            'footer_note' => null,
            'footer_website' => null,
            'footer_email' => null,
            'footer_phone' => null,
            'show_coach_info' => true,
            'show_coach_photo' => false,
            'show_coach_credentials' => true,
            'show_coach_contact' => true,
            'paper_size' => 'a4',
            'orientation' => 'portrait',
            'margin_top' => 20,
            'margin_bottom' => 20,
            'margin_left' => 15,
            'margin_right' => 15,
            'font_family' => 'DejaVu Sans',
            'font_size_base' => 11,
            'font_size_header' => 24,
            'custom_css' => null,
            'watermark_text' => null,
            'watermark_enabled' => false,
            'is_active' => true,
        ];

        // Type-specific defaults
        switch ($type) {
            case self::TYPE_INVOICE:
                $defaults['header_title'] = 'FACTURA';
                $defaults['primary_color'] = '#1E3A8A';
                break;
            case self::TYPE_WORKOUT:
                $defaults['header_title'] = 'RUTINA DE ENTRENAMIENTO';
                $defaults['primary_color'] = '#3B82F6';
                $defaults['secondary_color'] = '#F97316';
                break;
            case self::TYPE_MEAL_PLAN:
                $defaults['header_title'] = 'PLAN DE ALIMENTACIÓN';
                $defaults['primary_color'] = '#10B981';
                $defaults['secondary_color'] = '#F59E0B';
                break;
            case self::TYPE_STATISTICS:
                $defaults['header_title'] = 'ESTADÍSTICAS DE PROGRESO';
                $defaults['primary_color'] = '#8B5CF6';
                break;
        }

        return $defaults;
    }

    /**
     * Clear cache when saving
     */
    protected static function booted()
    {
        static::saved(function ($setting) {
            Cache::forget("pdf_settings_{$setting->document_type}");
        });

        static::deleted(function ($setting) {
            Cache::forget("pdf_settings_{$setting->document_type}");
        });
    }

    /**
     * Get computed margin string for DomPDF
     */
    public function getMarginStringAttribute(): string
    {
        return "{$this->margin_top}mm {$this->margin_right}mm {$this->margin_bottom}mm {$this->margin_left}mm";
    }

    /**
     * Get logo URL
     */
    public function getLogoUrlAttribute(): ?string
    {
        if (!$this->logo) {
            return null;
        }
        
        return asset('storage/' . $this->logo);
    }
}
