<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PushSubscription extends Model
{
    protected $fillable = [
        'user_id',
        'endpoint',
        'p256dh_key',
        'auth_token',
        'user_agent',
        'device_type',
        'is_active',
        'last_used_at',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'last_used_at' => 'datetime',
    ];

    /**
     * Get the user that owns this subscription.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope for active subscriptions.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get subscriptions for a specific user.
     */
    public static function forUser(int $userId)
    {
        return static::where('user_id', $userId)->active()->get();
    }

    /**
     * Update the last used timestamp.
     */
    public function updateLastUsed(): bool
    {
        $this->last_used_at = now();
        return $this->save();
    }

    /**
     * Deactivate this subscription.
     */
    public function deactivate(): bool
    {
        $this->is_active = false;
        return $this->save();
    }
}
