<?php

namespace App\Notifications;

use App\Models\Invoice;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InvoiceGenerated extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Invoice $invoice
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Factura Generada - ' . $this->invoice->invoice_number)
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('Se ha generado una nueva factura para tu suscripción.')
            ->line('')
            ->line('**Detalles de la Factura:**')
            ->line('Número: ' . $this->invoice->invoice_number)
            ->line('Total: $' . number_format($this->invoice->total, 2))
            ->line('Estado: ' . ucfirst($this->invoice->status))
            ->action('Ver Factura', route('coach.subscription.invoices'))
            ->line('Puedes descargar el PDF de tu factura desde tu panel de control.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'invoice_generated',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'total' => $this->invoice->total,
            'status' => $this->invoice->status,
            'message' => "📄 Factura {$this->invoice->invoice_number} generada por \${$this->invoice->total}",
            'title' => 'Nueva factura generada',
            'url' => route('coach.subscription.invoices'),
        ];
    }
}
