<?php

namespace App\Notifications;

use App\Models\User;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class NewCoachRegistered extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public User $coach
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Nuevo Coach Registrado - Pendiente de Aprobación')
            ->greeting('¡Hola Administrador!')
            ->line('Un nuevo coach se ha registrado en la plataforma y está pendiente de aprobación.')
            ->line('')
            ->line('**Detalles del Coach:**')
            ->line('Nombre: ' . $this->coach->name)
            ->line('Email: ' . $this->coach->email)
            ->line('Fecha de registro: ' . $this->coach->created_at->format('d/m/Y H:i'))
            ->action('Revisar y Aprobar', route('admin.users.show', $this->coach->id))
            ->line('Por favor revisa la información y aprueba o rechaza la solicitud.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'new_coach_registered',
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'coach_email' => $this->coach->email,
            'message' => "👤 Nuevo coach registrado: {$this->coach->name} ({$this->coach->email}) - Pendiente de aprobación",
            'title' => 'Nuevo coach registrado',
            'url' => route('admin.users.show', $this->coach->id),
        ];
    }
}
