<?php

namespace App\Notifications;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to customer when their order is delivered.
 */
class OrderDelivered extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Order $order
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Pedido Entregado - #' . $this->order->order_number)
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('Tu pedido ha sido entregado correctamente.')
            ->line('')
            ->line('**Detalles del Pedido:**')
            ->line('• Número de Pedido: ' . $this->order->order_number)
            ->line('• Fecha de Entrega: ' . $this->order->delivered_at->format('d/m/Y H:i'))
            ->line('• Total: ' . $this->order->formatCurrency($this->order->total))
            ->line('')
            ->line('Esperamos que disfrutes tu compra. Si tienes algún problema o comentario sobre los productos, no dudes en contactarnos.')
            ->action('Ver Historial de Pedidos', $this->getOrdersUrl($notifiable))
            ->line('')
            ->line('¡Gracias por comprar con nosotros!')
            ->salutation('Nos vemos pronto.');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'order_delivered',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'delivered_at' => $this->order->delivered_at?->toISOString(),
            'message' => "Tu pedido #{$this->order->order_number} ha sido entregado",
            'title' => 'Pedido Entregado',
            'url' => $this->getOrderUrl($notifiable),
        ];
    }

    protected function getOrderUrl(object $notifiable): string
    {
        $prefix = match($notifiable->role) {
            'admin' => 'admin',
            'coach' => 'coach',
            default => 'client',
        };

        return route("{$prefix}.store.orders.show", $this->order->id);
    }

    protected function getOrdersUrl(object $notifiable): string
    {
        $prefix = match($notifiable->role) {
            'admin' => 'admin',
            'coach' => 'coach',
            default => 'client',
        };

        return route("{$prefix}.store.orders.index");
    }

    public function databaseType(object $notifiable): string
    {
        return 'order_delivered';
    }
}
