<?php

namespace App\Notifications;

use App\Models\Invoice;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PaymentProblem extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Invoice $invoice,
        public string $errorMessage = ''
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->error()
            ->subject('Problema con tu Pago - Acción Requerida')
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('Hemos detectado un problema al procesar tu pago.')
            ->line('')
            ->line('**Detalles:**')
            ->line('Factura: ' . $this->invoice->invoice_number)
            ->line('Monto: $' . number_format($this->invoice->total, 2))
            ->when($this->errorMessage, function($mail) {
                return $mail->line('Error: ' . $this->errorMessage);
            })
            ->line('')
            ->line('Para evitar la suspensión de tu cuenta, por favor actualiza tu método de pago o contacta con soporte.')
            ->action('Actualizar Pago', route('coach.subscription.index'))
            ->line('Si tienes alguna pregunta, no dudes en contactarnos.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'payment_problem',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'amount' => $this->invoice->total,
            'error_message' => $this->errorMessage,
            'url' => route('coach.subscription.index'),
        ];
    }
}
