<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ReferralActive extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public User $referredUser,
        public float $bonusAmount = 0
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('¡Tu Referido está Activo!')
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('¡Excelentes noticias! Tu referido **' . $this->referredUser->name . '** ha activado su cuenta.')
            ->line('');

        if ($this->bonusAmount > 0) {
            $message->line('**Has ganado un bono de $' . number_format($this->bonusAmount, 2) . '**')
                ->line('El bono ha sido agregado a tu wallet y puedes usarlo para pagar tus próximas suscripciones.');
        }

        return $message
            ->action('Ver Mi Wallet', route('coach.wallet.index'))
            ->line('¡Sigue compartiendo tu código de referido y gana más beneficios!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'referral_active',
            'referred_user_id' => $this->referredUser->id,
            'referred_user_name' => $this->referredUser->name,
            'bonus_amount' => $this->bonusAmount,
            'url' => route('coach.wallet.index'),
        ];
    }
}
