<?php

namespace App\Notifications;

use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionRenewed extends Notification implements ShouldQueue
{
    use Queueable;

    protected Subscription $subscription;
    protected string $paymentMethod;
    protected float $amountPaid;

    public function __construct(Subscription $subscription, string $paymentMethod = 'card', float $amountPaid = 0)
    {
        $this->subscription = $subscription;
        $this->paymentMethod = $paymentMethod;
        $this->amountPaid = $amountPaid;
    }

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Tu suscripcion ha sido renovada exitosamente')
            ->greeting('Hola ' . $notifiable->name . '!')
            ->line('Tu suscripcion al plan **' . $this->subscription->plan->name . '** ha sido renovada exitosamente.');

        if ($this->amountPaid > 0) {
            $message->line('**Monto cobrado:** $' . number_format($this->amountPaid, 2));
        }

        $message->line('**Metodo de pago:** ' . $this->getPaymentMethodLabel())
            ->line('**Nueva fecha de vencimiento:** ' . $this->subscription->end_date->format('d/m/Y'));

        if ($this->subscription->isRecurring()) {
            $message->line('Tu suscripcion se renovara automaticamente el ' . $this->subscription->next_billing_date?->format('d/m/Y') . '.');
        }

        return $message
            ->action('Ver Mi Suscripcion', url('/coach/subscription'))
            ->line('Gracias por continuar confiando en CoachPro!');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'subscription_renewed',
            'subscription_id' => $this->subscription->id,
            'plan_name' => $this->subscription->plan->name,
            'end_date' => $this->subscription->end_date->toDateString(),
            'amount_paid' => $this->amountPaid,
            'payment_method' => $this->paymentMethod,
            'message' => 'Tu suscripcion al plan ' . $this->subscription->plan->name . ' ha sido renovada hasta el ' . $this->subscription->end_date->format('d/m/Y') . '.',
            'title' => 'Suscripcion renovada',
            'url' => route('coach.subscription.index'),
        ];
    }

    protected function getPaymentMethodLabel(): string
    {
        return match ($this->paymentMethod) {
            'stripe' => 'Tarjeta de credito/debito',
            'paypal' => 'PayPal',
            'bank_transfer' => 'Transferencia bancaria',
            'wallet' => 'Saldo de wallet',
            default => ucfirst($this->paymentMethod),
        };
    }
}
