<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\WalletTransaction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WalletWithdrawalApproved extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public WalletTransaction $transaction
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'DOP');
        $amount = number_format(abs($this->transaction->amount), 2);

        return (new MailMessage)
            ->subject("Retiro Aprobado - \${$amount} {$currency}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu solicitud de retiro ha sido aprobada.")
            ->line("**Monto:** \${$amount} {$currency}")
            ->line("El pago será procesado según los plazos establecidos.")
            ->action('Ver Mi Billetera', route('coach.wallet.index'))
            ->salutation("Saludos,\nEl equipo de {$siteName}");
    }

    public function toArray(object $notifiable): array
    {
        $amount = number_format(abs($this->transaction->amount), 2);

        return [
            'type' => 'wallet_withdrawal_approved',
            'transaction_id' => $this->transaction->id,
            'amount' => abs($this->transaction->amount),
            'message' => "✅ Retiro de \${$amount} aprobado",
            'title' => 'Retiro aprobado',
            'url' => route('coach.wallet.index'),
        ];
    }
}
