import { ref, watch, onMounted } from 'vue';

const isDark = ref(false);
const isInitialized = ref(false);

export function useDarkMode() {
    const initDarkMode = () => {
        if (isInitialized.value) return;
        
        // Check localStorage first
        const savedTheme = localStorage.getItem('theme');
        
        if (savedTheme) {
            isDark.value = savedTheme === 'dark';
        } else {
            // Check system preference
            isDark.value = window.matchMedia('(prefers-color-scheme: dark)').matches;
        }
        
        applyTheme();
        isInitialized.value = true;
        
        // Listen for system theme changes
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
            if (!localStorage.getItem('theme')) {
                isDark.value = e.matches;
                applyTheme();
            }
        });
    };
    
    const applyTheme = () => {
        if (isDark.value) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    };
    
    const toggleDarkMode = () => {
        isDark.value = !isDark.value;
        localStorage.setItem('theme', isDark.value ? 'dark' : 'light');
        applyTheme();
    };
    
    const setDarkMode = (value) => {
        isDark.value = value;
        localStorage.setItem('theme', value ? 'dark' : 'light');
        applyTheme();
    };
    
    // Watch for changes
    watch(isDark, () => {
        applyTheme();
    });
    
    // Initialize on mount
    onMounted(() => {
        initDarkMode();
    });
    
    return {
        isDark,
        toggleDarkMode,
        setDarkMode,
        initDarkMode
    };
}
