import { ref, onMounted, onUnmounted } from 'vue';

/**
 * Composable para gestionar funcionalidades PWA
 * @returns {Object} Estado y métodos PWA
 */
export function usePWA() {
    const isInstallable = ref(false);
    const isInstalled = ref(false);
    const isOnline = ref(navigator.onLine);
    const isStandalone = ref(false);

    // Verificar si está instalado como PWA
    const checkStandalone = () => {
        isStandalone.value = 
            window.matchMedia('(display-mode: standalone)').matches ||
            window.navigator.standalone === true ||
            document.referrer.includes('android-app://');
    };

    // Manejar evento de instalación disponible
    const handleInstallable = () => {
        isInstallable.value = true;
    };

    // Manejar cambios de conexión
    const handleOnline = () => {
        isOnline.value = true;
    };

    const handleOffline = () => {
        isOnline.value = false;
    };

    // Instalar PWA
    const install = async () => {
        if (window.installPWA) {
            const result = await window.installPWA();
            if (result) {
                isInstallable.value = false;
                isInstalled.value = true;
            }
            return result;
        }
        return false;
    };

    // Actualizar Service Worker
    const update = () => {
        if (window.updateSW) {
            window.updateSW(true);
        }
    };

    onMounted(() => {
        checkStandalone();
        
        // Verificar si ya es instalable
        if (window.isPWAInstallable && window.isPWAInstallable()) {
            isInstallable.value = true;
        }

        // Listeners
        window.addEventListener('pwa-installable', handleInstallable);
        window.addEventListener('online', handleOnline);
        window.addEventListener('offline', handleOffline);
        
        // Listener para cambio de display mode
        const mediaQuery = window.matchMedia('(display-mode: standalone)');
        mediaQuery.addEventListener('change', checkStandalone);
    });

    onUnmounted(() => {
        window.removeEventListener('pwa-installable', handleInstallable);
        window.removeEventListener('online', handleOnline);
        window.removeEventListener('offline', handleOffline);
    });

    return {
        isInstallable,
        isInstalled,
        isOnline,
        isStandalone,
        install,
        update,
    };
}

/**
 * Composable para detectar tipo de dispositivo
 * @returns {Object} Información del dispositivo
 */
export function useDevice() {
    const isMobile = ref(false);
    const isTablet = ref(false);
    const isDesktop = ref(false);
    const isTouchDevice = ref(false);
    const isIOS = ref(false);
    const isAndroid = ref(false);
    const screenWidth = ref(window.innerWidth);
    const screenHeight = ref(window.innerHeight);

    const detectDevice = () => {
        const ua = navigator.userAgent;
        
        // Detectar SO
        isIOS.value = /iPad|iPhone|iPod/.test(ua);
        isAndroid.value = /Android/.test(ua);
        
        // Detectar touch
        isTouchDevice.value = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
        
        // Detectar tamaño (Tailwind breakpoints)
        const width = window.innerWidth;
        screenWidth.value = width;
        screenHeight.value = window.innerHeight;
        
        isMobile.value = width < 768;
        isTablet.value = width >= 768 && width < 1024;
        isDesktop.value = width >= 1024;
    };

    const handleResize = () => {
        detectDevice();
    };

    onMounted(() => {
        detectDevice();
        window.addEventListener('resize', handleResize);
    });

    onUnmounted(() => {
        window.removeEventListener('resize', handleResize);
    });

    return {
        isMobile,
        isTablet,
        isDesktop,
        isTouchDevice,
        isIOS,
        isAndroid,
        screenWidth,
        screenHeight,
    };
}

/**
 * Composable para manejar gestos táctiles
 * @param {Object} options - Configuración de gestos
 * @returns {Object} Métodos y eventos de gestos
 */
export function useSwipe(options = {}) {
    const { 
        threshold = 50, 
        timeout = 500,
        onSwipeLeft = null,
        onSwipeRight = null,
        onSwipeUp = null,
        onSwipeDown = null,
    } = options;

    let touchStartX = 0;
    let touchStartY = 0;
    let touchStartTime = 0;

    const handleTouchStart = (e) => {
        touchStartX = e.touches[0].clientX;
        touchStartY = e.touches[0].clientY;
        touchStartTime = Date.now();
    };

    const handleTouchEnd = (e) => {
        const touchEndX = e.changedTouches[0].clientX;
        const touchEndY = e.changedTouches[0].clientY;
        const touchEndTime = Date.now();

        const diffX = touchEndX - touchStartX;
        const diffY = touchEndY - touchStartY;
        const timeDiff = touchEndTime - touchStartTime;

        // Solo procesar si está dentro del tiempo límite
        if (timeDiff > timeout) return;

        const absDiffX = Math.abs(diffX);
        const absDiffY = Math.abs(diffY);

        // Determinar dirección principal del swipe
        if (absDiffX > absDiffY && absDiffX > threshold) {
            if (diffX > 0 && onSwipeRight) {
                onSwipeRight();
            } else if (diffX < 0 && onSwipeLeft) {
                onSwipeLeft();
            }
        } else if (absDiffY > absDiffX && absDiffY > threshold) {
            if (diffY > 0 && onSwipeDown) {
                onSwipeDown();
            } else if (diffY < 0 && onSwipeUp) {
                onSwipeUp();
            }
        }
    };

    const bindSwipe = (element) => {
        if (element) {
            element.addEventListener('touchstart', handleTouchStart, { passive: true });
            element.addEventListener('touchend', handleTouchEnd, { passive: true });
        }
    };

    const unbindSwipe = (element) => {
        if (element) {
            element.removeEventListener('touchstart', handleTouchStart);
            element.removeEventListener('touchend', handleTouchEnd);
        }
    };

    return {
        bindSwipe,
        unbindSwipe,
        handleTouchStart,
        handleTouchEnd,
    };
}

/**
 * Composable para manejar pull-to-refresh
 * @param {Function} onRefresh - Callback al hacer refresh
 * @returns {Object} Estado y métodos
 */
export function usePullToRefresh(onRefresh) {
    const isRefreshing = ref(false);
    const pullDistance = ref(0);
    const threshold = 80;
    
    let startY = 0;
    let isPulling = false;

    const handleTouchStart = (e) => {
        if (window.scrollY === 0) {
            startY = e.touches[0].clientY;
            isPulling = true;
        }
    };

    const handleTouchMove = (e) => {
        if (!isPulling) return;
        
        const currentY = e.touches[0].clientY;
        const diff = currentY - startY;
        
        if (diff > 0 && window.scrollY === 0) {
            pullDistance.value = Math.min(diff * 0.5, threshold * 1.5);
        }
    };

    const handleTouchEnd = async () => {
        if (pullDistance.value >= threshold && onRefresh) {
            isRefreshing.value = true;
            await onRefresh();
            isRefreshing.value = false;
        }
        
        pullDistance.value = 0;
        isPulling = false;
    };

    const bindPullToRefresh = (element) => {
        if (element) {
            element.addEventListener('touchstart', handleTouchStart, { passive: true });
            element.addEventListener('touchmove', handleTouchMove, { passive: false });
            element.addEventListener('touchend', handleTouchEnd, { passive: true });
        }
    };

    const unbindPullToRefresh = (element) => {
        if (element) {
            element.removeEventListener('touchstart', handleTouchStart);
            element.removeEventListener('touchmove', handleTouchMove);
            element.removeEventListener('touchend', handleTouchEnd);
        }
    };

    return {
        isRefreshing,
        pullDistance,
        threshold,
        bindPullToRefresh,
        unbindPullToRefresh,
    };
}
